<?php

use support\Cache;

function IMTAG($toFunction = null){
    static $instances = [];
    static $methodParams = [];

    // 参数校验
    if (!$toFunction || !str_contains($toFunction, '@')) {
        throw new InvalidArgumentException('Invalid function format');
    }

    [$className, $methodName] = explode('@', $toFunction, 2);
    $fullClassName = "app\\common\\imtrait\\" . ucfirst($className);

    // 实例缓存逻辑
    if (!isset($instances[$className])) {
        if (!class_exists($fullClassName)) {
            throw new RuntimeException("Class {$fullClassName} not found");
        }
        $instances[$className] = new $fullClassName();
    }
    $instance = $instances[$className];

    // 方法校验
    if (!method_exists($instance, $methodName)) {
        throw new RuntimeException("Method {$methodName} not found");
    }

    // 参数缓存与处理
    if (!isset($methodParams[$toFunction])) {
        try {
            $reflection = new ReflectionMethod($instance, $methodName);
            $methodParams[$toFunction] = array_map(
                fn($param) => $param->isDefaultValueAvailable()
                    ? $param->getDefaultValue()
                    : null,
                $reflection->getParameters()
            );
        } catch (ReflectionException $e) {
            throw new RuntimeException("Reflection failed: " . $e->getMessage());
        }
    }

    // 合并参数
    $args = array_slice(func_get_args(), 1);
    $mergedParams = array_replace(
        $methodParams[$toFunction],
        array_combine(array_keys($methodParams[$toFunction]), $args)
    );

    return call_user_func_array([$instance, $methodName], $mergedParams);
}

/**
 * 获取网站配置
 * @return array
 */
function GetWebDb()
{
    if ($res = Cache::get("webdb")) {
        return $res;
    }

    $res = ['public' => [], 'private' => []];
    $_m = new \app\common\model\Config();
    $confList = $_m->GetList(['status' => '1','GetFile' => 'field,conf_value,private'],false);
    foreach ($confList as $v) {
        $key = empty($v['private']) ? 'public' : 'private';
        $res[$key][$v['field']] = $v['conf_value'];
    }
    $res = FileEditAdd($res, false);
    // 添加锁文件存在性检查和错误处理
    $lockFile = base_path() . 'app.lock';
    if (file_exists($lockFile)) {
        $lockContent = file_get_contents($lockFile);
        if ($lock = json_decode($lockContent, true)) {
            $res['public']['version_no'] = $lock['version_no'] ?? '';
            $res['public']['version_server'] = $lock['server'] ?? 'https://www.wormcms.com';
        }
    }
    // 添加默认值以防锁文件缺失
    $res['public']['version_no'] = $res['public']['version_no'] ?? 'v2';
    $res['public']['version_server'] = $res['public']['version_server'] ?? 'https://www.wormcms.com';
    Cache::set("webdb", $res, 86400);
    return $res;
}
function GetServer($data,$private = true){
    $res = $private ? [...$data['public'], ...$data['private']] : $data;
    $u = EmpowerUrl(request()->host());
    $targetUrl = $u['url'];
    $set = [
        'url' => $targetUrl,
        'pack' => 'tp',
        'empower' => '1',
        'version' => 'V4',
    ];
    $_ecode = GetEruSing(GetEruPwd($set,$targetUrl),$targetUrl);
    $res['web_footbq'] ??= '';
    $res['web_warrant'] ??= '';
    if($_ecode !== $res['web_warrant']){
        $_Y = date('Y');
        $res['web_footbq'] .= "<div class='ery-fex-l'>Powered by <a style='margin: 0 5px;' href='//wormcms.com' title='EruCMS' target='_blank'>EruCMS</a> © 2012-{$_Y}</div>";
    }
    return $res;
}
/**
 * 清空/删除 文件夹
 * @param string $dirname 文件夹路径
 * @param bool $self 是否删除当前文件夹
 * @return bool
 */
function DelDir($dirname, $self = true)
{
    if (!file_exists($dirname)) {
        return false;
    }
    if (is_file($dirname) || is_link($dirname)) {
        return unlink($dirname);
    }
    // 使用队列进行迭代遍历，避免递归
    $queue = [$dirname]; // 初始化队列，存储待处理的目录
    $dirsToRemove = [];  // 存储所有目录路径，稍后删除（从最深开始）
    while (!empty($queue)) {
        $currentDir = array_shift($queue); // 取出队列中的第一个目录
        if (!is_dir($currentDir) || is_link($currentDir)) {
            continue; // 跳过文件或符号链接（应在迭代中处理）
        }
        // 打开目录句柄
        if (($dirHandle = opendir($currentDir)) === false) {
            continue; // 跳过无法打开的目录（可选：记录错误）
        }
        // 遍历目录条目
        while (($entry = readdir($dirHandle)) !== false) {
            if ($entry === '.' || $entry === '..') {
                continue; // 跳过当前和父目录
            }
            $path = $currentDir . DIRECTORY_SEPARATOR . $entry; // 使用 DIRECTORY_SEPARATOR 提高兼容性
            if (is_dir($path) && !is_link($path)) {
                // 如果是目录（非符号链接），添加到队列处理
                array_push($queue, $path);
                $dirsToRemove[] = $path; // 标记为稍后删除
            } else {
                // 删除文件或符号链接
                unlink($path);
            }
        }
        closedir($dirHandle); // 关闭目录句柄
    }
    // 删除所有目录：从最深子目录开始（反向排序）
    rsort($dirsToRemove); // 按路径深度降序排序
    foreach ($dirsToRemove as $dir) {
        rmdir($dir);
    }
    // 根据 $self 删除根目录
    if ($self) {
        rmdir($dirname);
    }
    return true;
}
/**
 * @param $_dir 要读取的文件目录
 * @param array $getfile
 * @return array|mixed
 */
function GetDirlist(string $dirPath): array
{
    if (!is_dir($dirPath)) {
        return [];
    }
    $dirIterator = new FilesystemIterator($dirPath,FilesystemIterator::KEY_AS_FILENAME | FilesystemIterator::CURRENT_AS_PATHNAME | FilesystemIterator::SKIP_DOTS);
    $fileList = [];
    foreach ($dirIterator as $file) {
        $fileName = $dirIterator->getFilename();
        if ($fileName[0] === '.') {
            continue;
        }
        if ($fileName === '.DS_Store' || $fileName === 'Thumbs.db') {
            continue;
        }
        $fileList[] = $fileName;
    }
    return $fileList;
}
/**
 * 读取目录内所有文件
 * @param $_dir
 * @param array $getfile
 * @return array|mixed
 */
function ReadDirList($_dir){
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator(
            $_dir,
            FilesystemIterator::SKIP_DOTS |
            FilesystemIterator::UNIX_PATHS // 统一路径为 Unix 风格
        ),
        RecursiveIteratorIterator::SELF_FIRST
    );

    $result = [];
    foreach ($iterator as $item) {
        /** @var SplFileInfo $item */
        $result[] = [
            'path' => $item->getPathname(),
            'type' => $item->getType(),      // 文件类型：file/dir/link
            'size' => $item->getSize(),      // 文件大小（字节）
            'mtime' => $item->getMTime(),    // 最后修改时间（时间戳）
            'is_dir' => $item->isDir(),
            'is_file' => $item->isFile(),
            'extension' => $item->getExtension() // 文件扩展名
        ];
    }
    return $result;
}
/**
 * 复制文件
 * @param $old
 * @param $new
 * @return bool
 */
function CopyFiles(string $old, string $new): bool {
    $destinationDir = dirname($new);
    if (!is_dir($destinationDir) && !mkdir($destinationDir, 0755, true)) {
        return false;
    }
    return @copy($old, $new);
}
/**
 * 读取文件内容
 */
function GetReadFile($filename, $method = "rb")
{
    if ($handle = @fopen($filename, $method)) {
        @flock($handle, LOCK_SH);
        $filedata = @fread($handle, @filesize($filename));
        @fclose($handle);
    }
    return $filedata;
}
/**
 * 写入文件内容 (优化版本)
 */
function WriteFile(string $filename, $data, string $method = "wb", bool $lock = true): array
{
    @touch($filename);
    $handle = @fopen($filename,$method);
    if(!$handle){
        return ['code' => '0','msg' => $filename . '文件不可写，请检查权限'];
    }
    if($lock){
        @flock($handle,LOCK_EX);
    }
    @fputs($handle,$data);
    if($method == "rb+"){
        @ftruncate($handle,strlen($data));
    }
    @fclose($handle);
    @chmod($filename,0755);
    if(is_writable($filename) ){
        return ['code' => '1','msg' => '文件写入成功'];
    }else{
        return ['code' => '0','msg' => '文件不可写，请检查权限'];
    }
}
/**
 * 生成url地址
 */
function GetEruUrl($fun, $getdata = []) {
    // 直接返回有效URL
    if (filter_var($fun, FILTER_VALIDATE_URL)) {
        return $fun;
    }
    // 优化查询参数分割
    if (empty($getdata)) {
        if (($qmark_pos = strpos($fun, '?')) !== false) {
            $getdata = substr($fun, $qmark_pos + 1);
            $fun = substr($fun, 0, $qmark_pos);
        }
    }
    // 使用原生函数解析查询参数
    if (!empty($getdata) && !is_array($getdata)) {
        parse_str($getdata, $parsed);
        $getdata = $parsed;
    }
    // 优化路径格式转换判断
    if (substr_count($fun, '/') > 1) {
        $fun = strtr($fun, ['/' => '.']);
    }
    return route($fun,$getdata);
}
function EmpowerUrl(string $url): array
{
    $domain = parse_url($url, PHP_URL_HOST) ?: str_replace(['http://', 'https://'], '', $url);
    $segments = explode('.', trim($domain, '.'));
    $size = count($segments);
    if ($size > 2 && $segments[0] === 'www') {
        array_shift($segments);
        $size--;
    }
    $cleanDomain = implode('.', $segments);
    return [
        'url' => $cleanDomain,
        'url_size' => $size
    ];
}
/**
 * 数据集转数组
 * @param mixed $data
 * @return mixed
 */
function JobArray(mixed $data): mixed {
    if (is_array($data)) {
        if (!$data) return [];
        return match(true) {
            is_object($data[array_key_first($data)]) => array_map(
                static fn($v) => is_object($v) ? $v->toArray() : $v,
                $data
            ),
            default => $data
        };
    }
    return is_object($data) ? $data->toArray() : $data;
}
/**
 * 数组指定位置插入元素
 */
function ArraySpliceArray(array &$data, $index, $add)
{
    $c = count($data);//总长度
    if ($c <= $index) {
        $data[] = $add;
        return $data;
    };
    for ($i = ($c - 1); $i >= 0; $i--) {//循环转移数组元素
        $data[($i + 1)] = $data[$i];
        if ($index == $i) {
            $data[$i] = $add;
            break;
        }
    }
    return $data;
}
//  数组排序
function ArraySort($arr, $keys, $type = 'desc') {
    $keysvalue = $new_array = array();
    foreach ($arr as $k => $v){
        $keysvalue[$k] = empty($v[$keys]) ? '0' : $v[$keys];
    }
    $type == 'asc' ? asort($keysvalue) : arsort($keysvalue);
    reset($keysvalue);
    foreach ($keysvalue as $k => $v) {
        if(!is_array($arr[$k])){
            $arr[$k] = $arr[$k]->toArray();
        }
        $new_array[] = $arr[$k];
    }
    return $new_array;
}
/**
 * 格式化栏目列表，去除自身及下级栏目
 * @param array $data 要处理的数据
 * @param array $pid   要移除的下级ID,id为下级ID,on为是否移除自身
 * @param array $ToFile 指定递归字段，默认标题为title,值为id
 * @return array 返回处理好的数组
 */
function FormatPart($data,$pid = [],$ToFile = []){
    if (empty($data)){
        return $data;
    }
    $data = \app\common\hook\NodeFormat::toList($data);
    $ToFile['title'] = empty($ToFile['title']) ? 'title' : $ToFile['title'];
    $ToFile['value'] = empty($ToFile['value']) ? 'id' : $ToFile['value'];
    if(!empty($pid['id'])){
        $_ids = \app\common\hook\NodeFormat::getChildsId($data,$pid['id']);
        if(!empty($pid['on'])){
            array_push($_ids,$pid['id']);
        }
        $data = DelFile($data,$ToFile['value'],$_ids,true);
    }
    return SelectEdit($data,$ToFile);
}
/**
 * 格式化表单多选项，如下拉菜单，单选，多选
 */
function SelectEdit(array $data, array $files = []): array {
    if (!$data) {
        return $data;
    }
    $res = [];
    $titleKey = $files['title'] ?? 'title';
    $valueKey = $files['value'] ?? 'value';
    foreach ($data as $v) {
        $res[] = [
            'title' => $v[$titleKey],
            'value' => $v[$valueKey]
        ];
    }
    return $res;
}
/**
 * 提取内容图片列表
 * 如果$data为空，则返回空数组。
 * 使用正则表达式提取$data中的图片链接，并存储在$imgdata数组中。
 * 返回$imgdata数组中索引为1的元素，即提取的图片链接数组。
 * @param $data
 * @return array
 */
function GetImgList($data){
    if(empty($data)){
        return [];
    }
    $imgps = '/<img\b[^>]+?src=([\'"])(?<url>[^\'">]+)\1[^>]*>/i';
    preg_match_all($imgps, $data, $imgdata);
    return $imgdata['url'];
}
/**
 * 处理新老数据
 * @param $data
 * @param $old
 * @param $file
 * @return array
 */
function CompareFile($data,$old,$file){
    $data = $data ?? [];
    $old = $old ?? [];
    // 一次性提取文件值数组
    $newFiles = $data ? array_column($data, $file) : [];
    $oldFiles = $old ? array_column($old, $file) : [];
    // 使用索引查找替代in_array
    $oldFileLookup = array_flip($oldFiles);
    $newdata = [];
    foreach ($data as $item) {
        $fileValue = $item[$file] ?? null;
        if (!$fileValue || isset($oldFileLookup[$fileValue])){
            continue;
        }
        $newdata[] = $item;
    }
    // 反转角色避免二次循环
    $newFileLookup = array_flip($newFiles);
    $oldToRemove = [];
    foreach ($old as $item) {
        $fileValue = $item[$file] ?? null;
        if (!$fileValue || isset($newFileLookup[$fileValue])){
            continue;
        }
        $oldToRemove[] = $item;
    }
    return [
        'new' => $newdata,
        'old' => $oldToRemove
    ];
}
/**
 * 递归移除所有空元素（包括空字符串、空白字符串、空数组）
 * @param array $data 输入数组（支持多维数组）
 * @return array 处理后的非空数组
 */
function DelNull(array $data): array
{
    return array_reduce(
        array_keys($data),
        function (array $carry, $key) use ($data) {
            $value = $data[$key];
            // 递归处理子数组
            if (is_array($value)) {
                $processed = DelNull($value);
                if (!empty($processed)) {
                    $carry[$key] = $processed;
                }
                return $carry;
            }
            // 处理标量值
            $strVal = (string)$value;
            if (trim($strVal) !== '') {
                $carry[$key] = $strVal;
            }
            return $carry;
        },
        []
    );
}
/**
 * 格式化数据
 * @array $data 要处理的数据
 * @return array 处理完毕的数据
 */
function DataTrim($data) {
    $trim = [];
    foreach ($data as $key => $val) {
        if (is_array($val)) {
            $trim[$key] = DataTrim($val);
        } else {
            // 统一处理空值并转为字符串
            $val = (string) ($val ?? '');
            // 合并空格处理流程
            $val = trim($val);
            if ($val !== '') {
                // 合并制表符和空格替换为单次正则替换
                $val = preg_replace(
                    ['/\t/', '/\s{3,}/'],
                    ['   ', ' '],
                    $val);
                $val = trim($val);
            }
            // 简化空值判断
            $trim[$key] = $val !== '' ? $val : null;
        }
    }
    return $trim;
}
/**
 * 去除不需要的字段
 * @param array $data 要处理的数组
 * @param string $file 处理字段
 * @param array|string $value 对比值
 * @param bool $eqfile 等于为true
 * @return array mixed 返回处理完成的数组
 */
function DelFile($data, $file, $value, $eqfile = false) {
    // 如果值不是数组，转换为数组统一处理逻辑
    $values = is_array($value) ? $value : [$value];
    $result = [];
    foreach ($data as $item) {
        // 统一转换为数组处理
        $item = is_array($item) ? $item : (array)$item;
        // 检查字段是否存在
        if (!isset($item[$file])) {
            continue;
        }
        $fieldValue = $item[$file];
        // 检查值是否在搜索值列表中
        $inValues = in_array($fieldValue, $values);
        // 确定是否保留该项
        if (($eqfile && !$inValues) || (!$eqfile && $inValues)) {
            $result[] = $item;
        }
    }
    return $result;
}
function DelArrayFile(array $data, array $fields, bool $deleteMode = false)
{
    $fieldDict = array_flip($fields);
    if ($deleteMode) {
        return array_diff_key($data, $fieldDict);
    }
    return array_intersect_key($data, $fieldDict);
}
/**
 * 替换文件路径中的上传目录前缀（支持数组递归处理）
 * @param mixed $data 输入数据（字符串或数组）
 * @param bool $isAdd 是否为添加模式（true: 替换为外部URL，false: 还原为本地路径）
 * @param bool $api 是否处理API模式（涉及数据库配置）
 * @return array|string
 */
function FileEditAdd($data, bool $isAdd = true, bool $api = false)
{
    if (empty($data)) {
        return $data;
    }
    // 递归处理数组
    if (is_array($data)) {
        return array_map(function ($item) use ($isAdd, $api) {
            return FileEditAdd($item, $isAdd, $api);
        }, $data);
    }
    $data = (string) $data;
    $webUrl = '';

    // 获取API模式下的网站URL（避免重复查询）
    if ($api) {
        $webUrl = (new \app\common\model\Config())->whereField('web_url')->value('conf_value');
    }

    // 定义替换规则对
    $replacements = $isAdd ? [
        '/upload_file/' => 'http://www_cxbs_net/Ls_dir/',
        $webUrl . '/upload_file/' => 'http://www_cxbs_net/Ls_dir/' // API模式反向替换
    ] : [
        'http://www_cxbs_net/Ls_dir/' => '/upload_file/',
        '/upload_file/' => $webUrl . '/upload_file/' // API模式正向替换
    ];
    // 按优先级执行替换
    foreach ($replacements as $search => $replace) {
        if ($search === null) continue; // 跳过未初始化的规则
        $data = str_replace($search, $replace, $data);
    }
    return $data;
}
/**
 * 删除文件
 * @param string $dirname 文件路径
 * @return bool
 */
function DelWebFile($data){
    if(!empty($data) && is_array($data)){
        foreach ($data as $key => $val){
            DelWebFile($val);
        }
    }else if(!empty($data) && is_file(public_path().$data)){
        $filedir = explode('/upload_file',$data);
        if(count($filedir) > '1'){
            @unlink(public_path().$data);
        }
    }
    return true;
}
/**
 * 加密解密
 * @string $data 要加密的数据
 * @string $key 密钥
 * @param bool $decrypt 是否解密
 * @return string
 */
function GetEruPwd($data,$key,$decrypt = false){
    if(!is_array($key)){
        $key = [
            'key' => $key
        ];
        $key['iv'] = substr(md5($key['key']),0,16);
    } else {
        if(empty($key['key']) || empty(key(['iv']))){
            return false;
        }
    }
    if($decrypt){
        $res = openssl_decrypt($data,'AES-128-CBC',$key['key'],0,$key['iv']);
        $res = json_decode($res,true);
    } else {
        $res = openssl_encrypt(json_encode($data,JSON_UNESCAPED_UNICODE),'AES-128-CBC',$key['key'],0,$key['iv']);
    }
    return $res;
}
function GetEruSing($data, $key): string
{
    static $ipadPattern = null, $opadPattern = null;
    $ipadPattern ??= str_repeat("\x36", 64);  // 十六进制字面量替代chr()
    $opadPattern ??= str_repeat("\x5c", 64);  // 使用\x表示法提高效率
    // 单行键处理：截取+填充
    $key = str_pad(substr($key, 0, 64), 64, "\0");
    // 直接异或计算
    $innerHash = md5(($key ^ $ipadPattern) . $data, true);

    return strtoupper(md5(($key ^ $opadPattern) . $innerHash));
}
/**
 * 下划线转驼峰
 * @param string $str 字符串
 * @param bool $ucfirst 首字母是否大写
 * @param string $sep 分隔符
 * @return string | array
 */
function ToUnderScore($str,$tolower = true,$sep = '_'){
    if(is_array($str)){
        foreach ($str as $k => $v){
            $str[$k] = ToUnderScore($v,$tolower,$sep);
        }
        return $str;
    }
    if($tolower){
        $str = strtolower(preg_replace('/([a-z])([A-Z])/', "$1" . $sep . "$2", $str));
    }else{
        $str = array_filter(explode($sep, $str));
        foreach ($str as $k => $v){
            $str[$k] = trim(ucfirst($v));
        }
        $str = implode('',$str);
    }
    return $str;
}
/**
 * @param string $msg 响应提示
 * @param string $code 响应码
 * @param string|array $data  返回数据
 * @param string $url 返回链接
 * @param string $wait  等待时间
 * @param false $json   返回数据格式
 * @param string $temp 使用模板
 * @param int $browser_code 浏览器响应码
 * @throws Exception
 */
function ResClient($msg = '查询成功',$code = '1',$data = '',$url = '',$wait = '3',$json = false,$temp = '',$browser_code = 200){
    if(!empty(request()->viewconf['app']) && request()->viewconf['app'] == 'api'){
        $json = true;
    }
    $res = [
        'code' => $code,
        'msg' => $msg,
        'wait' => $wait ?? '3',
        'time' => time(),
        'url' => $url,
        'json' => $json,
        'temp' => empty($temp) ? '' : $temp,
        'data' => $data
    ];
    throw new \app\common\support\EruException(json_encode($res,JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES),$browser_code);
}
/**
 * @param string $msg 响应提示
 * @param string $code 响应码
 * @param string|array $data  返回数据
 * @param string $url 返回链接
 * @param false $json   返回数据格式
 * */
function ResJson($msg = '查询成功',$code = '1',$data = [],$url = ''){
    ResClient($msg,$code,$data,$url,'3',true);
}
/**
 * 响应重定向
 * @param $url
 * @param string $code
 * @param string $msg
 */
function ResRedirect($url,$wait = '3',$browser_code = '302',$msg = '即将跳转'){
    ResClient($msg,'1','',$url,$wait,false,'jump.htm',$browser_code);
}
/**
 * 查询验证码是否正确
 * @param string $code
 * @return bool
 */
function CheckCaptcha($code){
    $captcha = request()->session()->pull('captcha');
    return $captcha == $code;
}
/** 大小单位转换  **/
function CountSize($size)
{
    $KB = 1024;
    $MB = 1024 * $KB;
    $GB = 1024 * $MB;
    $TB = 1024 * $GB;
    if ($size < $KB) {
        return $size . "B";
    } elseif ($size < $MB) {
        return round($size / $KB, 2) . " KB";
    } elseif ($size < $GB) {
        return round($size / $MB, 2) . " MB";
    } elseif ($size < $TB) {
        return round($size / $GB, 2) . " GB";
    } else {
        return round($size / $TB, 2) . " TB";
    }
}
/**
 * 产生随机字符串，不长于32位
 * @param int $length
 * @return 产生的随机字符串
 */
function GetNonceStr($length = 32): string
{
    $chars = "abcdefghijklmnopqrstuvwxyz0123456789";
    $str = "";
    for ($i = 0; $i < $length; $i++) {
        $str .= substr($chars, mt_rand(0, strlen($chars) - 1), 1);
    }
    return $str;
}
/**
 * 生成随机字符
 */
function GetRange($endnum = '8',$numbers = []){
    $numbers = empty($numbers) ? range (1,50) : $numbers;
    shuffle ($numbers);
    $num = 8;
    $result = array_slice($numbers,0,$num);
    $result = implode('',$result);
    $newname = '';
    for ( $i = 0; $i < $endnum; $i++ ) {
        $newname .= substr($result, mt_rand(0, strlen($result) - 1), 1);
    }
    return $newname;
}
/**
 * 截取字符串
 * @param string $string
 * @param int $length
 * @param int $more
 * @param string $dot
 * @return string
 */
function GetWordNum(string $string,$length,$more = 1,$dot = '..'){
    // 优化点1：前置边界检查 + 处理dot设置
    if (!$more) $dot = '';
    if (mb_strwidth($string) <= $length) {
        return $string;
    }
    // 优化点2：预定义实体映射提高替换效率
    $entityMap = [
        '&amp;' => "\x01&\x01",
        '&quot;' => "\x01\"\x01",
        '&lt;' => "\x01<\x01",
        '&gt;' => "\x01>\x01"
    ];
    $reversedMap = array_flip($entityMap);
    // 优化点3：单次替换操作
    $protected = strtr($string, $entityMap);
    // 优化点4：使用mb_strimwidth代替字节手动计算
    $truncated = mb_strimwidth($protected, 0, $length, '', 'UTF-8');
    // 优化点5：单次反向替换
    $result = strtr($truncated, $reversedMap);
    // 优化点6：不需要chr(1)扫描处理
    return $result . $dot;
}
/**
 * 10进制数转换成62进制数
 *
 * @param string $num
 * @return string
 */
function TO62($num) {
    $_to = 62;
    $_dict = 'abcdefghijklmnopqrstuvwxyz0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $_dict = str_split($_dict);
    $res = '';
    do {
        $res = $_dict[bcmod($num, $_to)] . $res;
        $num = bcdiv($num, $_to);
    } while ($num > 0);
    return $res;
}
/**
 * 62进制数转换成十进制数
 *
 * @param string $num
 * @return string
 */
function TO10($num) {
    $from = 62;
    $num = strval($num);
    $dict = 'abcdefghijklmnopqrstuvwxyz0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $len = strlen($num);
    $dec = 0;
    for($i = 0; $i < $len; $i++) {
        $pos = strpos($dict, $num[$i]);
        $dec = bcadd(bcmul(bcpow((string)$from, (string)($len - $i - 1)), (string) $pos), (string)$dec);
    }
    return $dec;
}
/**
 * 发送CURL请求
 * * @param string $url 请求URL
 * * @param array|string $post_data POST数据
 * * @param array|null $headers 请求头
 * * @param string $method HTTP方法 (GET|POST|PUT|DELETE等)
 * * @param int $timeout 超时时间(秒)
 * * @param bool $verifySSL 是否验证SSL证书
 * * @return array|string 返回包含状态码、响应头和响应体的数组
 * * @throws RuntimeException 当请求失败时抛出异常
 */
function RequestCurl(string $url = '',$post_data = [],?array $headers = null,string $method = 'POST',int $timeout = 30,bool $verifySSL = false) {
    // 验证URL
    if (empty($url) || !filter_var($url, FILTER_VALIDATE_URL)) {
        return ['code' => '-1','msg' => '无效的URL'];
    }
    // 初始化cURL
    $curl = curl_init();
    if ($curl === false) {
        return ['code' => '-1','msg' => '无法初始化CURL'];
    }
    // 设置基本选项
    $options = [
        CURLOPT_URL => $url,
        CURLOPT_CUSTOMREQUEST => strtoupper($method),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HEADER => true,
        CURLOPT_FAILONERROR => false,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_SSL_VERIFYPEER => $verifySSL,
        CURLOPT_SSL_VERIFYHOST => $verifySSL ? 2 : 0,
    ];
    // 设置请求头
    if (!empty($headers)) {
        $options[CURLOPT_HTTPHEADER] = $headers;
    }
    // 设置POST数据
    if (!empty($post_data) && in_array($method, ['POST', 'PUT', 'PATCH'])) {
        $options[CURLOPT_POSTFIELDS] = is_array($post_data) ? http_build_query($post_data) : $post_data;
    }
    curl_setopt_array($curl, $options);
    // 执行请求
    $response = curl_exec($curl);
    if ($response === false) {
        $error = curl_error($curl);
        curl_close($curl);
        return ['code' => '-1','msg' => "cURL请求失败: {$error}"];
    }
    // 获取响应信息
    $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    $headerSize = curl_getinfo($curl, CURLINFO_HEADER_SIZE);
    $responseHeaders = substr($response, 0, $headerSize);
    $responseBody = substr($response, $headerSize);
    // 关闭连接
    curl_close($curl);
    // 解析响应头
    $headers = [];
    foreach (explode("\r\n", $responseHeaders) as $header) {
        if (strpos($header, ':') !== false) {
            list($key, $value) = explode(':', $header, 2);
            $headers[trim($key)] = trim($value);
        }
    }

    return $responseBody;
}
